"use client";
import { useState } from "react";
import { Loader2, Lock } from "lucide-react";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";

interface ChangePasswordDialogProps {
  isOpen: boolean;
  technicianEmail: string;
  technicianName?: string | null;
  onClose: () => void;
  onSave: (newPassword: string) => Promise<void>;
}

export default function ChangePasswordDialog({
  isOpen,
  technicianEmail,
  technicianName,
  onClose,
  onSave,
}: ChangePasswordDialogProps) {
  const { t } = useLanguage();
  const [newPassword, setNewPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const handleSave = async () => {
    setError(null);

    if (!newPassword) {
      setError(t("admin.passwordRequired"));
      return;
    }

    if (newPassword.length < 6) {
      setError(t("admin.passwordMinChars"));
      return;
    }

    if (newPassword !== confirmPassword) {
      setError(t("admin.passwordsDontMatch"));
      return;
    }

    setSaving(true);
    try {
      await onSave(newPassword);
      // Reset form on success
      setNewPassword("");
      setConfirmPassword("");
      onClose();
    } catch (error: any) {
      setError(error.message || "Failed to change password");
    } finally {
      setSaving(false);
    }
  };

  const handleClose = () => {
    setNewPassword("");
    setConfirmPassword("");
    setError(null);
    onClose();
  };

  if (!isOpen) return null;

  return (
    <Dialog
      open={isOpen}
      onOpenChange={(open) => !open && handleClose()}
      modal={true}
    >
      <DialogContent className='sm:max-w-[600px]'>
        <DialogHeader>
          <DialogTitle>{t("admin.changePassword")}</DialogTitle>
        </DialogHeader>

        <div className='grid gap-4 py-4'>
          <div className='rounded-lg bg-blue-50 p-4 border border-blue-200'>
            <div className='flex items-start gap-3'>
              <Lock className='w-5 h-5 text-blue-600 mt-0.5' />
              <div className='flex-1'>
                <p className='text-sm font-medium text-blue-900 mb-1'>
                  {t("admin.changePasswordFor")}
                </p>
                <p className='text-sm text-blue-700'>
                  {technicianName || "—"} ({technicianEmail})
                </p>
              </div>
            </div>
          </div>

          {error && (
            <div className='rounded-lg bg-red-50 p-3 border border-red-200'>
              <p className='text-sm text-red-700'>{error}</p>
            </div>
          )}

          <div className='grid gap-2'>
            <Label htmlFor='newPassword'>{t("admin.newPassword")}</Label>
            <Input
              id='newPassword'
              type='password'
              value={newPassword}
              onChange={(e) => setNewPassword(e.target.value)}
              placeholder={t("admin.enterNewPassword") || "Enter new password"}
              minLength={6}
              disabled={saving}
            />
            <p className='text-xs text-gray-500'>
              {t("admin.passwordMinChars")}
            </p>
          </div>

          <div className='grid gap-2'>
            <Label htmlFor='confirmPassword'>
              {t("admin.confirmPassword")}
            </Label>
            <Input
              id='confirmPassword'
              type='password'
              value={confirmPassword}
              onChange={(e) => setConfirmPassword(e.target.value)}
              placeholder={t("admin.confirmNewPassword") || "Confirm new password"}
              disabled={saving}
            />
          </div>
        </div>

        <div className='flex justify-end gap-3'>
          <Button variant='outline' onClick={handleClose} disabled={saving}>
            {t("common.cancel")}
          </Button>
          <Button onClick={handleSave} disabled={saving}>
            {saving ? (
              <>
                <Loader2 className='mr-2 h-4 w-4 animate-spin' />
                {t("admin.changing")}
              </>
            ) : (
              t("admin.changePassword")
            )}
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}
